from __future__ import annotations

import threading
import tkinter as tk
from tkinter import messagebox, scrolledtext, ttk
from typing import Callable, Dict, Optional

try:
    from ttkbootstrap import Style  # type: ignore
except ImportError:  # pragma: no cover - optional dependency
    Style = None  # type: ignore

from . import cleanup, power, reporting, services, system_info, tweaks, utils


class OptimizerGUI:
    def __init__(self) -> None:
        self.root = tk.Tk()
        self.root.title("FZ-M1 Optimizer")
        self.root.geometry("900x600")
        self.root.minsize(780, 520)

        if Style:
            self.style = Style(theme="flatly")
        else:
            self.style = None

        self.log_messages: list[str] = []
        self._build_layout()
        self._populate_diagnosis()

    def run(self) -> None:
        self.root.mainloop()

    def _build_layout(self) -> None:
        self.status_var = tk.StringVar(value="Gata de optimizare.")
        self.admin_var = tk.StringVar(
            value="Mod administrator" if utils.is_running_as_admin() else "Mod normal"
        )

        header_frame = ttk.Frame(self.root, padding=10)
        header_frame.pack(fill="x")

        ttk.Label(header_frame, text="FZ-M1 Optimizer", font=("Segoe UI", 18)).pack(
            side="left"
        )
        ttk.Label(header_frame, textvariable=self.admin_var, padding=(10, 0)).pack(
            side="right"
        )

        notebook = ttk.Notebook(self.root)
        notebook.pack(expand=True, fill="both", padx=10, pady=(0, 10))
        self.notebook = notebook

        self.frames: Dict[str, ttk.Frame] = {}
        for name in ["Diagnoză", "Curățare", "Servicii", "Power", "Tweaks", "Log"]:
            frame = ttk.Frame(notebook, padding=12)
            notebook.add(frame, text=name)
            self.frames[name] = frame

        self._build_diag_tab()
        self._build_cleanup_tab()
        self._build_services_tab()
        self._build_power_tab()
        self._build_tweaks_tab()
        self._build_log_tab()

        status_bar = ttk.Frame(self.root, padding=(10, 5))
        status_bar.pack(fill="x")
        ttk.Label(status_bar, textvariable=self.status_var).pack(side="left")

    def _build_diag_tab(self) -> None:
        frame = self.frames["Diagnoză"]
        self.sys_info_text = scrolledtext.ScrolledText(frame, height=20, state="disabled")
        self.sys_info_text.pack(expand=True, fill="both")

        refresh_btn = ttk.Button(frame, text="Reîncarcă informațiile", command=self._populate_diagnosis)
        refresh_btn.pack(pady=8)

    def _build_cleanup_tab(self) -> None:
        frame = self.frames["Curățare"]
        ttk.Label(frame, text="Operațiuni de curățare", font=("Segoe UI", 12, "bold")).pack(
            anchor="w"
        )

        actions = [
            ("Șterge fișiere temporare", cleanup.clean_temp_directories),
            ("Curăță cache browser", cleanup.cleanup_browser_caches),
            ("Curăță Prefetch", cleanup.purge_prefetch),
            ("Curăță cache Windows Update", cleanup.clear_windows_update_cache),
            ("Rulează Storage Sense", cleanup.run_storage_sense),
        ]

        for text, func in actions:
            ttk.Button(frame, text=text, command=lambda f=func, t=text: self._run_async(f, t)).pack(
                anchor="w", pady=4
            )

    def _build_services_tab(self) -> None:
        frame = self.frames["Servicii"]
        ttk.Label(frame, text="Status servicii", font=("Segoe UI", 12, "bold")).pack(anchor="w")
        ttk.Button(frame, text="Actualizează listă servicii", command=self._refresh_services).pack(
            anchor="w", pady=(0, 10)
        )
        self.services_tree = ttk.Treeview(frame, columns=("Categorie", "Nume"), show="headings")
        self.services_tree.heading("Categorie", text="Categorie")
        self.services_tree.heading("Nume", text="Nume serviciu")
        self.services_tree.pack(expand=True, fill="both")

        action_frame = ttk.Frame(frame)
        action_frame.pack(fill="x", pady=10)

        ttk.Label(action_frame, text="Nume serviciu:").grid(row=0, column=0, sticky="w")
        self.service_name_var = tk.StringVar()
        ttk.Entry(action_frame, textvariable=self.service_name_var, width=40).grid(
            row=0, column=1, padx=5
        )

        ttk.Button(
            action_frame,
            text="Pornește",
            command=lambda: self._service_state("start"),
        ).grid(row=0, column=2, padx=5)
        ttk.Button(
            action_frame,
            text="Oprește",
            command=lambda: self._service_state("stop"),
        ).grid(row=0, column=3, padx=5)

        ttk.Label(action_frame, text="Setare pornire:").grid(row=1, column=0, sticky="w", pady=6)
        self.start_mode_var = tk.StringVar(value="demand")
        ttk.Combobox(
            action_frame,
            textvariable=self.start_mode_var,
            values=["auto", "demand", "disabled"],
            width=10,
        ).grid(row=1, column=1, sticky="w")
        ttk.Button(
            action_frame,
            text="Aplică",
            command=self._configure_service,
        ).grid(row=1, column=2, padx=5)

    def _build_power_tab(self) -> None:
        frame = self.frames["Power"]
        ttk.Label(frame, text="Planuri de alimentare", font=("Segoe UI", 12, "bold")).pack(
            anchor="w"
        )
        ttk.Button(frame, text="Actualizează planuri", command=self._refresh_power_plans).pack(
            anchor="w", pady=(0, 10)
        )
        self.power_plans_tree = ttk.Treeview(
            frame, columns=("GUID", "Nume"), show="headings", height=8
        )
        self.power_plans_tree.heading("GUID", text="GUID")
        self.power_plans_tree.heading("Nume", text="Nume plan")
        self.power_plans_tree.pack(expand=True, fill="both")

        actions_frame = ttk.Frame(frame)
        actions_frame.pack(fill="x", pady=10)

        ttk.Button(
            actions_frame,
            text="Creează plan Power Saver optimizat",
            command=self._create_power_plan,
        ).grid(row=0, column=0, padx=5)
        ttk.Button(
            actions_frame,
            text="Aplică plan selectat",
            command=self._set_power_plan,
        ).grid(row=0, column=1, padx=5)
        ttk.Button(
            actions_frame,
            text="Dezactivează hibernarea",
            command=lambda: self._run_async(power.disable_hibernation, "Dezactivează hibernare"),
        ).grid(row=0, column=2, padx=5)

    def _build_tweaks_tab(self) -> None:
        frame = self.frames["Tweaks"]
        ttk.Label(frame, text="Tweaks sistem", font=("Segoe UI", 12, "bold")).pack(anchor="w")
        ttk.Button(
            frame,
            text="Setează efecte vizuale minim",
            command=lambda: self._run_async(tweaks.apply_visual_effects_minimal, "Tweaks vizuale"),
        ).pack(anchor="w", pady=4)
        ttk.Button(
            frame,
            text="Dezactivează aplicații în fundal",
            command=lambda: self._run_async(tweaks.disable_background_apps, "Dezactivează background"),
        ).pack(anchor="w", pady=4)
        ttk.Button(
            frame,
            text="Minimizează telemetria",
            command=lambda: self._run_async(tweaks.disable_telemetry, "Telemetry tweak"),
        ).pack(anchor="w", pady=4)

    def _build_log_tab(self) -> None:
        frame = self.frames["Log"]
        self.log_text = scrolledtext.ScrolledText(frame, state="disabled")
        self.log_text.pack(expand=True, fill="both")
        ttk.Button(
            frame,
            text="Salvează logul",
            command=self._save_log,
        ).pack(pady=8)

    def _update_status(self, message: str) -> None:
        self.status_var.set(message)
        self.root.after(4000, lambda: self.status_var.set("Gata de optimizare."))

    def _append_log(self, message: str) -> None:
        self.log_messages.append(message)
        self.log_text.configure(state="normal")
        self.log_text.insert("end", f"{message}\n")
        self.log_text.configure(state="disabled")
        self.log_text.see("end")

    def _run_async(self, func: Callable, label: str) -> None:
        def worker() -> None:
            try:
                result = func()
                self.root.after(0, lambda: self._append_log(f"{label}: {result}"))
            except Exception as exc:
                self.root.after(
                    0, lambda: self._append_log(f"Eroare la {label}: {exc}")
                )
            finally:
                self.root.after(0, lambda: self._update_status(f"{label} finalizat."))

        threading.Thread(target=worker, daemon=True).start()
        self._update_status(f"{label} în lucru...")

    def _populate_diagnosis(self) -> None:
        info = system_info.collect_system_overview()
        processes = system_info.list_top_processes()
        services_summary = services.get_services_snapshot()

        lines = [
            "=== Sistem ===",
            f"Nume computer: {info.computer_name}",
            f"OS: {info.os_version}",
            f"Arhitectură: {info.architecture}",
            f"CPU: {info.cpu}",
            f"Nuclee fizice/logice: {info.physical_cores}/{info.logical_cores}",
            f"Frecvență bază (GHz): {info.base_clock_ghz}",
            f"RAM total/disponibil (GB): {info.total_ram_gb}/{info.available_ram_gb}",
            f"Uptime: {info.uptime}",
            f"Spațiu liber C: {info.system_drive_free_gb} GB",
            "",
            "=== Procese consumatoare ===",
        ]
        for proc in processes:
            lines.append(
                f"{proc.name} (PID {proc.pid}) - CPU {proc.cpu_percent}% - RAM {proc.memory_usage_mb} MB"
            )

        lines.append("")
        lines.append("=== Servicii ===")
        for key, values in services_summary.items():
            lines.append(f"{key.title()}: {len(values)} servicii")

        text = "\n".join(lines)
        self.sys_info_text.configure(state="normal")
        self.sys_info_text.delete("1.0", "end")
        self.sys_info_text.insert("end", text)
        self.sys_info_text.configure(state="disabled")
        self._append_log("Diagnoză actualizată.")

    def _refresh_services(self) -> None:
        snapshot = services.get_services_snapshot()
        for item in self.services_tree.get_children():
            self.services_tree.delete(item)
        for category, service_list in snapshot.items():
            for svc in service_list:
                self.services_tree.insert("", "end", values=(category, svc))
        self._append_log("Serviciile au fost actualizate.")

    def _service_state(self, action: str) -> None:
        name = self.service_name_var.get().strip()
        if not name:
            messagebox.showwarning("Serviciu", "Introdu numele serviciului (nume intern).")
            return

        def worker() -> None:
            try:
                output = services.change_service_state(name, action)
                self.root.after(
                    0, lambda: self._append_log(f"Serviciu {name} {action}: {output}")
                )
            except Exception as exc:
                self.root.after(0, lambda: self._append_log(f"Eroare {action} {name}: {exc}"))

        threading.Thread(target=worker, daemon=True).start()

    def _configure_service(self) -> None:
        name = self.service_name_var.get().strip()
        if not name:
            messagebox.showwarning("Serviciu", "Introdu numele serviciului (nume intern).")
            return
        start_mode = self.start_mode_var.get()

        def worker() -> None:
            try:
                output = services.configure_service_start_mode(name, start_mode)
                self.root.after(
                    0, lambda: self._append_log(f"Serviciu {name} -> {start_mode}: {output}")
                )
            except Exception as exc:
                self.root.after(
                    0, lambda: self._append_log(f"Eroare configurare {name}: {exc}")
                )

        threading.Thread(target=worker, daemon=True).start()

    def _refresh_power_plans(self) -> None:
        plans = power.list_power_plans()
        for item in self.power_plans_tree.get_children():
            self.power_plans_tree.delete(item)

        for guid, name in plans.items():
            self.power_plans_tree.insert("", "end", values=(guid, name))
        self._append_log("Planurile de alimentare au fost actualizate.")

    def _create_power_plan(self) -> None:
        def worker() -> None:
            try:
                guid = power.duplicate_power_plan("power_saver", "FZ-M1 Saver")
                power.configure_power_settings(guid)
                power.set_active_plan(guid)
                self.root.after(
                    0,
                    lambda: self._append_log(
                        f"Planul 'FZ-M1 Saver' creat și activat (GUID {guid})."
                    ),
                )
                self.root.after(0, self._refresh_power_plans)
            except Exception as exc:
                self.root.after(0, lambda: self._append_log(f"Eroare plan alimentare: {exc}"))

        threading.Thread(target=worker, daemon=True).start()

    def _set_power_plan(self) -> None:
        selection = self.power_plans_tree.selection()
        if not selection:
            messagebox.showinfo("Plan alimentare", "Selectează un plan din listă.")
            return
        guid = self.power_plans_tree.item(selection[0])["values"][0]

        def worker() -> None:
            try:
                power.set_active_plan(guid)
                self.root.after(0, lambda: self._append_log(f"Planul {guid} a fost activat."))
            except Exception as exc:
                self.root.after(0, lambda: self._append_log(f"Eroare activare plan: {exc}"))

        threading.Thread(target=worker, daemon=True).start()

    def _save_log(self) -> None:
        content = "\n".join(self.log_messages)
        path = reporting.write_text_log("fz_m1_optimizer", content)
        messagebox.showinfo("Log salvat", f"Logul a fost salvat în {path}")


def launch_app() -> None:
    app = OptimizerGUI()
    app.run()

